# Copyright 2022 Wireflow AB

# Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

# 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

# 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

# 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import minimalmodbus
import serial

class ProgrammableResistor(minimalmodbus.Instrument):
    
    #the initialization of the device and the serial port parameters
    def __init__(self, port):
        minimalmodbus.Instrument.__init__(self,port,1)
        self.mode=minimalmodbus.MODE_RTU
        self.serial.baudrate=115200
        self.serial.bytesize=8
        self.serial.parity=minimalmodbus.serial.PARITY_EVEN
    
    #the clear function to close the serial port manually
    def clear(self):
        self.serial.close()
    
    #accessing the register that contain the serial number of the device
    def get_serial_number(self):
        return self.read_register(4,0,4)
    
    #accessing the register that contain the device type number
    def get_device_type(self):
        return self.read_register(8,0,4)
    
    #accessing the registers that contain the calibration date
    #and formatting it to a string (year-month-date)
    def get_calibration_date(self):
        date=self.read_registers(5,3,4)
        return ('-'.join(str(d) for d in date))
    
    #accessing the registers that contain the firmaware version data of the device
    #and formatting it to a string (n.n.n.nnn)
    def get_firmware_version(self):
        firm=self.read_registers(0,4,4)
        return ('.'.join(str(f) for f in firm))
    
    #this is the function that writes to the registers
    #it takes as arguments the channel from which the registers are calculated and the value to be written
    def set_resistance_one_chan(self,channel,value):
        if(type(value)!=float and type(value)!=int):
            raise InputTypeError
        elif(type(channel)!=int):
            raise InputTypeError
        else:
            self.write_float(channel*2,float(value),2,0)
    
    #this is the function that reads the value that the selected channel(register) has
    #it takes as arguments the channel which we want to know the value to
    def get_resistance_output_one_chan(self,channel):
        if(type(channel)!=int):
            raise InputTypeError
        else:
            return(self.read_float(9+channel*2,4,2,0))
    
    #this is the function that writes to the grouped channels registers
    #it takes as arguments the channel group name and the value to be written
    #the registers are calculated according to the channel group name
    def set_resistance_one_chan_enhanced(self,channelgroup,value):
        if(type(value)!=int and type(value)!=float):
            raise InputTypeError
        elif (channelgroup=='0-1'):
            self.write_float(8,float(value),2,0)
        elif (channelgroup=='2-3'):
            self.write_float(10,float(value),2,0)
        else:
            raise GroupError()
    
    #the function writes to all the registers of all the 4 individual channels
    #it takes as arguments a list of values that we want to write to each of the channel
    #the value position gives the channel that the value will be set to
    def set_resistance_four_chan(self,values):
        if(len(values)!=4):
            raise ListError
        else:
            for i in range(len(values)):
                if(type(values[i])!=int and type(values[i])!=float):
                    raise InputTypeError
                else:
                    self.write_float(i*2,float(values[i]),2,0)
    
    #the function reads all 4(four) channels and gets the values saved in those registers
    #it does not take any arguments
    def get_resistance_output_four_chan(self):
        res=[]
        for i in range(4):
            res.append(self.read_float(9+i*2,4,2,0))
        return res

class GroupError(Exception):
    def __init__(self):
        self.msg="Wrong channel group"
        super().__init__(self.msg)

class ListError(Exception):
    def __init__(self):
        self.msg="The list should include 4 arguments"
        super().__init__(self.msg)

class InputTypeError(Exception):
    def __init__(self):
        self.msg="The value(s) should be a number (integer or float)"
        super().__init__(self.msg)
